# Code-First Migration TODO Guide
## Behavioral Preservation Migration for Complex Business Applications

### Target Configuration
```yaml
# FILL IN YOUR TARGET DETAILS BEFORE STARTING
source_language: "C#"           # Source language
source_framework: "ASP.NET Core" # Source framework
target_language: "Java"          # Target language  
target_framework: "Spring Boot"  # Target framework
target_version: "3.4.x"         # Framework version
database: "PostgreSQL"          # Target database
build_tool: "Gradle"            # Build system
```

### Migration Philosophy
This migration approach prioritizes **behavioral preservation** over artifact generation. We analyze source code directly to capture business logic, domain knowledge, and operational patterns that static specifications miss.

---

## TODO 1: Environment Setup & Repository Analysis
**Goal**: Understand the source codebase structure and set up target environment

### 1.1 Source Code Inventory
- [ ] **Analyze project structure** - identify main packages/namespaces
- [ ] **List all controllers/endpoints** - extract HTTP routes and methods
- [ ] **Catalog domain entities** - identify all business objects and their methods
- [ ] **Map service layer** - identify business logic services and their dependencies
- [ ] **Document data layer** - ORM configurations, repositories, database contexts
- [ ] **Identify external integrations** - APIs, message queues, third-party services
- [ ] **List middleware/filters** - authentication, logging, validation, custom middleware
- [ ] **🚨 CRITICAL: Identify middleware order** and dependencies (execution sequence matters)
- [ ] **🚨 CRITICAL: Extract middleware configurations** (when to apply, exclusions, conditional logic)

### 1.2 OpenAPI Contract Analysis & Generation
- [ ] **Check for existing OpenAPI specification** in source repository (openapi.yml/yaml/json)
- [ ] **If OpenAPI exists**: Extract and analyze for completeness vs actual controllers
- [ ] **If no OpenAPI**: Generate from controller analysis with exact method signatures
- [ ] **Extract all endpoint details**:
  - HTTP methods, paths, parameters (path, query, body, headers)
  - Request/response schemas with exact property names and types
  - Authentication/authorization requirements per endpoint
  - Error response codes and schemas
- [ ] **Create `docs/openapi.yml`** with complete API contract
- [ ] **Validate OpenAPI completeness** against controller inventory

### 1.3 System Architecture Visualization
- [ ] **Create `docs/diagrams.md`** with PlantUML diagrams for visual understanding
- [ ] **Generate Component Diagram**: Overall system architecture with services and dependencies
- [ ] **Create Database Schema Diagram**: All tables, relationships, and constraints
- [ ] **Draw Domain Entity Diagram**: All business entities with relationships and key methods
- [ ] **Document Integration Flow Diagram**: External systems and their interactions
- [ ] **Create Authentication Flow Diagram**: Login, token validation, and authorization flow

### 1.4 Target Environment Setup
- [ ] **Create target project** with chosen framework and build tool
- [ ] **Set up basic project structure** (controllers, services, entities, config)
- [ ] **Configure database connection** and basic ORM setup
- [ ] **Add essential dependencies** (web, security, database, testing, OpenAPI)
- [ ] **Copy `docs/openapi.yml`** to target project for implementation reference
- [ ] **Create placeholder README** with setup instructions

---

## TODO 2: Database Schema & Data Layer Migration
**Goal**: Establish data foundation with exact schema preservation

### 2.1 Schema Analysis & Migration
- [ ] **Extract current database schema** (tables, indexes, constraints, relationships)
- [ ] **Document foreign key relationships** and cascade behaviors
- [ ] **Identify stored procedures/functions** if any
- [ ] **Update Database Schema Diagram** in `docs/diagrams.md` with detailed schema
- [ ] **Create migration scripts** using target framework's migration tool
- [ ] **Set up entity classes** matching exact database schema
- [ ] **Configure ORM mappings** (JPA/Hibernate, Entity Framework, etc.)
- [ ] **Test database connectivity** and basic CRUD operations

### 2.2 Data Access Pattern Analysis
- [ ] **Map repository patterns** from source code
- [ ] **Identify transaction scopes** and isolation levels
- [ ] **Document query patterns** and performance optimizations
- [ ] **Extract connection pooling** and retry configurations
- [ ] **Implement equivalent data access layer** in target framework

---

## TODO 3: Domain Entity Migration (Business Logic Core)
**Goal**: Migrate all domain entities with exact business logic preservation

### 3.1 Entity Method Analysis
- [ ] **For each domain entity**, extract ALL public methods and their logic
- [ ] **Document business rules** embedded in entity methods
- [ ] **Map domain events** and when they're triggered
- [ ] **Identify validation rules** and constraints
- [ ] **Extract state transition logic** if any
- [ ] **🚨 CRITICAL: Extract hierarchical propagation logic** (e.g., C# `SetReplications` that propagates to ancestors/descendants)
- [ ] **🚨 CRITICAL: Identify special constant values** used in business logic (hardcoded UUIDs, enum values, configuration keys)
- [ ] **🚨 CRITICAL: Extract complex helper methods** that manipulate domain data (parsing, formatting, validation)

### 3.2 Entity Implementation
- [ ] **Create entity classes** with same properties and relationships
- [ ] **Implement ALL business methods** with exact same logic
- [ ] **Add validation rules** matching source implementation
- [ ] **Implement domain events** if used in source
- [ ] **Update Domain Entity Diagram** in `docs/diagrams.md` with implementation details
- [ ] **Create unit tests** for each entity method
- [ ] **Verify business rule compliance** through testing

---

## TODO 4: Service Layer Migration (Business Workflows)
**Goal**: Implement business services with exact workflow preservation

### 4.1 Service Analysis
- [ ] **For each service class**, map all public methods
- [ ] **Document external API calls** (URLs, timeouts, retry logic)
- [ ] **Extract transaction boundaries** and error handling
- [ ] **Map dependency injection** patterns
- [ ] **Identify caching strategies** and cache invalidation
- [ ] **Document async/background processing** patterns
- [ ] **🚨 CRITICAL: Extract conditional business logic** (special handling for specific IDs, parent relationships)
- [ ] **🚨 CRITICAL: Identify integration event publishing** (when events are triggered, what data is published)
- [ ] **🚨 CRITICAL: Document audit/publication tracking** (what gets logged, when, with what data)

### 4.2 Service Implementation  
- [ ] **Create service interfaces** matching source contracts
- [ ] **Implement ALL service methods** with exact business logic
- [ ] **Configure dependency injection** for all services
- [ ] **Add transaction management** matching source scopes
- [ ] **Implement error handling** with same exception types
- [ ] **Create Sequence Diagrams** in `docs/diagrams.md` for complex workflows
- [ ] **Add integration tests** for service workflows
- [ ] **Configure external API clients** with same timeouts/retries

---

## TODO 5: Authentication & Security Migration
**Goal**: Preserve exact authentication behavior and security policies

### 5.1 Authentication Analysis
- [ ] **Document authentication flow** (cookies, JWT, sessions, hybrid)
- [ ] **Extract authorization policies** and role mappings
- [ ] **Map token validation logic** and refresh mechanisms
- [ ] **Identify session management** strategies
- [ ] **Document OIDC/OAuth configuration** if used

### 5.2 Security Implementation
- [ ] **Configure authentication providers** matching source behavior
- [ ] **Implement authorization policies** with same access rules
- [ ] **Set up token validation** with exact same logic
- [ ] **Configure session management** if applicable
- [ ] **Update Authentication Flow Diagram** in `docs/diagrams.md` with implementation
- [ ] **Add security middleware** in correct order
- [ ] **Test authentication flows** end-to-end
- [ ] **Verify authorization behavior** for all endpoints

---

## TODO 6: Controller & API Layer Migration
**Goal**: Implement exact HTTP API contracts and behaviors

### 6.1 Controller Analysis
- [ ] **For each controller**, extract ALL endpoints with exact signatures
- [ ] **Document parameter binding** (path, query, body, headers)
- [ ] **Map response types** and status codes
- [ ] **Extract custom middleware** applied to endpoints
- [ ] **Document content type handling** and serialization

### 6.2 Controller Implementation
- [ ] **Create controller classes** with exact same endpoints
- [ ] **Implement parameter binding** matching source behavior
- [ ] **Add validation attributes** for request parameters
- [ ] **Configure response serialization** to match source format
- [ ] **Apply security attributes** to protected endpoints
- [ ] **Implement OpenAPI generation** to validate contract compliance
- [ ] **Add integration tests** for all endpoints
- [ ] **Verify HTTP contracts** match exactly with `docs/openapi.yml`
- [ ] **Update Component Diagram** in `docs/diagrams.md` with API layer

---

## TODO 7: Message Publishing & Integration Events
**Goal**: Preserve exact message publishing behavior and event handling

### 7.1 Messaging Analysis
- [ ] **Map all integration events** and their trigger conditions
- [ ] **Document message publishing patterns** (direct, outbox, etc.)
- [ ] **Extract queue/topic configurations** and routing keys
- [ ] **Identify event handlers** and their processing logic
- [ ] **Document retry and error handling** for messaging

### 7.2 Messaging Implementation
- [ ] **Configure message broker** (RabbitMQ, Kafka, etc.)
- [ ] **Implement integration events** matching source schemas
- [ ] **Set up message publishing** with same reliability patterns
- [ ] **Create event handlers** with exact same processing logic
- [ ] **Configure retries and dead letter** handling
- [ ] **Update Integration Flow Diagram** in `docs/diagrams.md` with messaging
- [ ] **Test message flows** end-to-end
- [ ] **Verify event ordering** and reliability

---

## TODO 8: External Integration Migration
**Goal**: Maintain exact external system integration behavior

### 8.1 Integration Analysis
- [ ] **List ALL external APIs** called by the application
- [ ] **Document API endpoints, headers, and payloads** used
- [ ] **Extract timeout and retry configurations** for each integration
- [ ] **Map authentication methods** for external services
- [ ] **Document error handling** for external failures
- [ ] **🚨 CRITICAL: Analyze parsing/formatting utilities** (e.g., X-Road client string parsing - extract FIRST segment, not second)
- [ ] **🚨 CRITICAL: Document format validation logic** (regex patterns, expected formats, error conditions)
- [ ] **🚨 CRITICAL: Extract environment-specific configurations** (dev vs prod URL differences, certificate handling)

### 8.2 Integration Implementation
- [ ] **Configure HTTP clients** for each external service
- [ ] **Set timeouts and retries** matching source configuration
- [ ] **Implement authentication** for external APIs
- [ ] **Add circuit breakers** if used in source
- [ ] **Implement error handling** with same fallback logic
- [ ] **Create integration tests** with mocked external services
- [ ] **Test failure scenarios** and error handling

---

## TODO 9: Configuration & Environment Migration
**Goal**: Replicate configuration management and environment setup

### 9.1 Configuration Analysis
- [ ] **Extract ALL configuration keys** and their purposes
- [ ] **Document environment-specific** values and defaults
- [ ] **Map configuration sources** (files, environment variables, secrets)
- [ ] **Identify configuration validation** rules
- [ ] **Document feature flags** if used

### 9.2 Configuration Implementation
- [ ] **Set up configuration management** using target framework patterns
- [ ] **Create configuration classes** with validation
- [ ] **Map environment variables** from source naming
- [ ] **Configure different environments** (dev, test, prod)
- [ ] **Add configuration validation** on startup
- [ ] **Test configuration loading** in different environments

---

## TODO 10: Logging & Observability Migration
**Goal**: Maintain operational visibility and debugging capabilities

### 10.1 Logging Analysis
- [ ] **Document logging patterns** and structured logging format
- [ ] **Extract correlation ID** handling and context propagation
- [ ] **Map log levels** and what triggers each level
- [ ] **Identify audit logging** requirements and patterns
- [ ] **Document metrics and monitoring** hooks

### 10.2 Observability Implementation
- [ ] **Configure structured logging** matching source format
- [ ] **Implement correlation ID** propagation
- [ ] **Add audit logging** with same events and structure
- [ ] **Set up metrics collection** for key operations
- [ ] **Configure health checks** for dependencies
- [ ] **Test logging and monitoring** in different scenarios

---

## TODO 11: Testing & Quality Assurance
**Goal**: Ensure behavioral parity through comprehensive testing

### 11.1 Test Migration Philosophy & Strategy

**🎯 Why Port Tests from Source Codebase?**

**✅ STRONGLY RECOMMENDED** - Porting tests is a **critical migration practice** for these reasons:

1. **Tests are Living Documentation**
   - Tests capture **actual business requirements** that documentation often misses
   - They show **edge cases** and **error scenarios** that specs don't cover
   - They demonstrate **expected behavior** in concrete, executable form

2. **Tests Reveal Hidden Business Logic**
   - **Real Example from our migration**: C# tests showed that empty replications lists should **remove replications from descendants**
   - **Without the test**: We would have missed this critical hierarchical behavior
   - **Impact**: Silent data inconsistency in production

3. **Tests Catch Framework/Language Differences**
   - **Real Example**: C# filters "openid" scope from JWT claims, Java doesn't by default
   - **Without the test**: Authentication behavior would differ between systems
   - **Impact**: Different permission sets, security inconsistencies

4. **Tests Validate Complex Integrations**
   - **Real Example**: Tests revealed that `QueuesSetIntegrationEvent` is only published for specific parent IDs
   - **Without the test**: Missing integration events for critical business entities
   - **Impact**: Downstream systems not notified of important changes

**🚫 Common Anti-Pattern: "We'll write new tests"**
- **Problem**: New tests reflect **your understanding**, not **source behavior**
- **Risk**: You'll miss nuanced business rules embedded in original tests
- **Result**: False confidence with incomplete behavioral coverage

**✅ Recommended Approach: "Port + Enhance"**
1. **Port all meaningful tests** from source codebase first
2. **Adapt to target framework** conventions and patterns
3. **Enhance with additional scenarios** specific to target platform
4. **Add performance/load tests** if missing from source

**🔄 Proven Test Migration Methodology (Based on Real Experience)**

**Phase 1: Integration-First Approach (Week 1-2)**
```
1. Port critical business workflow tests as API-level integration tests
2. Focus on happy path scenarios first
3. Use real database, real services (TestContainers)
4. Ignore unit test details initially
5. Goal: Prove core business logic works end-to-end
```

**Phase 2: Edge Case Coverage (Week 3)**
```
1. Port error handling and validation tests
2. Add boundary condition tests
3. Test failure scenarios and recovery
4. Focus on business rule violations
5. Goal: Ensure robust error handling
```

**Phase 3: Unit Test Refinement (Week 4)**
```
1. Add unit tests ONLY for complex algorithms
2. Mock external dependencies for isolated testing
3. Focus on pure business logic, not framework code
4. Skip unit tests for simple CRUD operations
5. Goal: Fast feedback for complex logic changes
```

**💰 ROI of Test Porting vs. Writing New Tests**

| Aspect | Port Existing Tests | Write New Tests | Winner |
|--------|-------------------|-----------------|---------|
| **Time Investment** | 2-3x faster (adapt vs create) | 5-10x slower (analyze + write) | 🏆 **Port** |
| **Business Logic Coverage** | Captures actual behavior | Captures your understanding | 🏆 **Port** |
| **Edge Case Discovery** | Reveals unknown edge cases | Misses subtle edge cases | 🏆 **Port** |
| **Regression Prevention** | Prevents behavioral drift | May miss critical behaviors | 🏆 **Port** |
| **Documentation Value** | Living docs of actual system | Docs of intended system | 🏆 **Port** |
| **Risk Reduction** | Lower risk of missing behaviors | Higher risk of gaps | 🏆 **Port** |

**📊 Real Migration Data:**
- **Tests Ported**: 8 domain tests + 8 integration tests = 16 tests
- **Critical Issues Found**: 7 major behavioral differences caught by tests
- **Time to Port**: ~4 hours
- **Estimated Time to Write New**: ~20+ hours
- **Production Issues Prevented**: Multiple silent data corruption scenarios

**🎯 When NOT to Port Tests:**
- **Framework-specific tests** (e.g., C# middleware registration tests)
- **Deprecated functionality** that won't exist in target
- **Performance tests** that are environment-specific
- **Flaky tests** that depend on external timing/services

### 11.2 Test Analysis & Requirements Extraction

**🎯 Goal: Understand WHAT to test, not HOW to copy tests**

- [ ] **Analyze existing test cases** from source codebase (don't copy structure!)
- [ ] **Extract business requirements** from each test - what behavior is being validated?
- [ ] **Document expected outcomes** for each scenario (success/failure conditions)
- [ ] **Map test scenarios** to actual user workflows and API operations
- [ ] **Identify edge cases** and boundary conditions tested
- [ ] **Extract performance benchmarks** if available
- [ ] **🚨 CRITICAL: Analyze test setup methods** - what reference data is created?
- [ ] **🚨 CRITICAL: Document test helper methods** - what business logic do they bypass?
- [ ] **🚨 CRITICAL: Extract assertion patterns** - what specific behaviors are validated?
- [ ] **🚨 CRITICAL: Identify business rules** hidden in test logic rather than production code
- [ ] **🚨 CRITICAL: Map test categories** - unit vs integration vs contract tests

### 11.3 Test Porting Best Practices

**🔄 Step-by-Step Test Migration Process:**

1. **Inventory Source Tests**
   - **Domain/Entity Tests**: Business logic, validation rules, state transitions
   - **Service Tests**: Workflows, integration patterns, error handling
   - **Controller Tests**: HTTP contracts, parameter binding, response formats
   - **Integration Tests**: End-to-end flows, external system interactions

2. **Categorize by Migration Priority**
   - **🔴 CRITICAL**: Domain logic tests (business rules, hierarchical operations)
   - **🟡 HIGH**: Service workflow tests (complex business processes)
   - **🟢 MEDIUM**: Controller contract tests (API behavior validation)
   - **🔵 LOW**: Infrastructure tests (framework-specific patterns)

3. **Port with Framework Adaptation**
   - **Preserve test intent** - what behavior is being validated?
   - **Adapt test structure** - use target framework's testing patterns
   - **Update assertions** - use target framework's assertion libraries
   - **Modernize test data** - use target framework's fixture patterns

4. **🎯 CRITICAL: Prefer Integration Over Unit Tests During Migration**
   
   **Why Integration Tests Are Superior for Migration:**
   
   | **Aspect** | **Unit Tests** | **Integration Tests** | **Winner** |
   |------------|----------------|----------------------|------------|
   | **Business Logic Validation** | Tests isolated components | Tests complete workflows | 🏆 **Integration** |
   | **Framework Integration** | May miss wiring issues | Catches configuration problems | 🏆 **Integration** |
   | **Database Schema Issues** | Uses mocks/stubs | Tests real constraints | 🏆 **Integration** |
   | **Cross-Service Communication** | Mocked dependencies | Real service interactions | 🏆 **Integration** |
   | **Regression Detection** | Limited scope | Full behavior validation | 🏆 **Integration** |
   | **Migration Risk Reduction** | High risk of gaps | Lower risk, more comprehensive | 🏆 **Integration** |
   
   **🔍 Real Migration Evidence from Our Experience:**
   - **Unit tests passed** but business logic was broken (tests bypassed validation)
   - **Integration tests caught** database constraint violations, foreign key issues
   - **Integration tests revealed** missing configuration and wiring problems
   - **Integration tests validated** complete user workflows, not just isolated methods
   - **Integration tests detected** retry mechanism interactions and error wrapping
   
   **📋 Migration Test Strategy Recommendation:**
   ```
   1. Start with API-level integration tests (80% of effort)
   2. Add focused unit tests for complex algorithms (20% of effort)
   3. Skip unit tests for simple CRUD operations
   4. Focus unit tests on pure business logic only
   ```

**🚨 Critical Testing Anti-Patterns to Avoid During Migration:**

| **Anti-Pattern** | **What We Did Wrong** | **Correct Approach** | **Why It Matters** |
|------------------|----------------------|---------------------|-------------------|
| **Bypassing Business Logic** | `addCodeDirectly()` helper methods | Call `service.create()` / `service.update()` | Unit tests passed but validation wasn't tested |
| **Testing Implementation, Not Behavior** | Test database inserts directly | Test business operations through API | Miss business rule violations |
| **Ignoring Error Message Changes** | Expect exact C# error messages | Check error message intent, handle wrapping | Retry mechanisms wrap original exceptions |
| **Using Fake Foreign Keys** | `UUID.randomUUID()` for references | Create actual referenced entities | Avoid foreign key constraint violations |
| **Incomplete Test Setup** | Missing parent entities | Create full entity hierarchy | Database constraints require complete data |
| **Framework-Specific Assumptions** | Assume same validation timing | Understand target framework lifecycle | Validation may happen at different points |

**🛠️ Common Porting Challenges & Solutions:**

| Challenge | C# Example | Java Solution | Key Insight |
|-----------|------------|---------------|-------------|
| **Mocking Frameworks** | `Mock<IService>` | `@Mock IService` | Different syntax, same concept |
| **Test Data Builders** | `FakeKodifikaatorBuilder` | `TestDataFactory` | Adapt builder pattern to target conventions |
| **Assertion Libraries** | `Assert.True(condition)` | `assertThat(condition).isTrue()` | Use target's fluent assertions |
| **Database Testing** | `InMemoryDatabase` | `@TestContainers` | Use target's database testing approach |
| **HTTP Testing** | `TestServer` | `MockMvc` | Adapt to target's web testing framework |
| **Exception Handling** | Direct business exceptions | Handle retry wrapper exceptions | Check both wrapper and cause messages |

**⚠️ Test Porting Pitfalls to Avoid:**

1. **Don't Over-Adapt Early**
   - **Wrong**: Immediately rewrite tests in "target style"
   - **Right**: Port first to capture behavior, then refactor for style

2. **Don't Skip "Boring" Tests**
   - **Wrong**: "This test just checks a simple getter"
   - **Right**: Simple tests often reveal important business rules

3. **Don't Assume Test Helpers are Equivalent**
   - **Wrong**: "Both create test data, they're the same"
   - **Right**: Compare what data is created, what constraints are enforced

4. **Don't Port Flaky Tests As-Is**
   - **Wrong**: Port timing-dependent or environment-specific tests unchanged
   - **Right**: Identify the core behavior being tested, stabilize in target

### 11.4 Test Implementation Strategy

**🎯 CRITICAL: Integration-First Approach (Proven Method)**

**❌ DON'T: Copy Unit Tests Directly**
```
❌ Copy C# unit test structure
❌ Port test helpers as-is  
❌ Test database operations directly
❌ Mock everything for "faster" tests
```

**✅ DO: Use Unit Tests as Examples, Write Integration Tests**
```
✅ Extract business requirements from unit tests
✅ Write integration tests for complete workflows
✅ Test through service layer (business logic)
✅ Use real infrastructure (database, containers)
```

**Phase 1: Analyze & Design (Don't Copy Yet!)**
- [ ] **Extract business requirements** from source unit tests
- [ ] **Identify user workflows** that trigger the tested behavior  
- [ ] **Map test scenarios** to API endpoints and service methods
- [ ] **Document expected behaviors** without implementation details

**Phase 2: Integration Test Implementation (80% of effort)**
- [ ] **Write API-level integration tests** for all business workflows
- [ ] **Use real database and services** (TestContainers approach)
- [ ] **Test complete user journeys** from request to response
- [ ] **Validate business rules** through service layer calls
- [ ] **Handle error scenarios** with proper exception expectations

**Phase 3: Focused Unit Tests (20% of effort, only where needed)**
- [ ] **Add unit tests** ONLY for complex algorithms or calculations
- [ ] **Mock external dependencies** for isolated testing  
- [ ] **Skip unit tests** for simple CRUD operations
- [ ] **Focus on pure business logic** without framework dependencies

**Phase 4: Validation & Refinement**
- [ ] **Run behavioral comparison** tests where possible
- [ ] **Achieve same business coverage** as source (not same test count)
- [ ] **Add performance tests** if missing from source
- [ ] **Create contract tests** for external integrations

**🚨 CRITICAL SUCCESS FACTORS (Lessons from Our Experience):**
- [ ] **Tests call actual business logic** (service methods, not helper methods)
- [ ] **Complete entity hierarchies** created for foreign key constraints  
- [ ] **Framework differences handled** (error messages, validation timing, retry wrappers)
- [ ] **Real infrastructure used** (database, message queues, external services)
- [ ] **Set up all required reference data** in test database (parent records, constants)
- [ ] **Use production-equivalent container configurations** in TestContainers
- [ ] **Include all required fields** in test command objects (`validFrom`, audit fields)
- [ ] **Test empty/null collection scenarios** that trigger business logic
- [ ] **Validate JWT claims processing** matches source filtering behavior

---

## TODO 12: Deployment & Infrastructure
**Goal**: Replicate deployment patterns and operational characteristics

### 12.1 Infrastructure Analysis
- [ ] **Document containerization** requirements and configurations
- [ ] **Extract deployment patterns** and environment setup
- [ ] **Map resource requirements** (CPU, memory, storage)
- [ ] **Document startup and shutdown** procedures
- [ ] **Identify monitoring and alerting** requirements

### 12.2 Infrastructure Implementation
- [ ] **Create container images** with same characteristics
- [ ] **Set up deployment scripts** for different environments
- [ ] **Configure resource limits** matching source requirements
- [ ] **Implement health checks** and readiness probes
- [ ] **Set up monitoring and alerting** rules
- [ ] **Test deployment procedures** in non-production environments

---

## TODO 13: Critical Gap Detection & Validation
**Goal**: Identify migration gaps before they become production issues

### 13.1 Deep Code Comparison
- [ ] **🚨 CRITICAL: Line-by-line entity method comparison** with source code
- [ ] **🚨 CRITICAL: Compare constants and configuration values** (UUIDs, enum values, hardcoded strings)
- [ ] **🚨 CRITICAL: Validate parsing/formatting logic** (string manipulation, date formatting, number parsing)
- [ ] **🚨 CRITICAL: Cross-reference utility methods** and helper classes
- [ ] **🚨 CRITICAL: Verify conditional logic branches** (if/else, switch statements, ternary operators)

### 13.2 Integration Point Validation  
- [ ] **🚨 CRITICAL: Test external API integration behavior** with real/mock services
- [ ] **🚨 CRITICAL: Validate middleware execution order** and conditional application
- [ ] **🚨 CRITICAL: Verify message publishing and event handling** exactly matches source
- [ ] **🚨 CRITICAL: Test error handling scenarios** (network failures, validation errors, database issues)
- [ ] **🚨 CRITICAL: Validate security behavior** (authentication, authorization, token handling)

### 13.3 Data Flow Verification
- [ ] **🚨 CRITICAL: Trace data transformations** from API input to database storage
- [ ] **🚨 CRITICAL: Verify hierarchical operations** (parent-child relationships, propagation logic)
- [ ] **🚨 CRITICAL: Validate audit logging** (what gets logged, when, with what data)
- [ ] **🚨 CRITICAL: Test edge cases** identified in source code comments or documentation

---

## TODO 14: Final Validation & Behavioral Verification
**Goal**: Confirm complete behavioral parity with source system

### 13.1 Comprehensive Testing
- [ ] **Run side-by-side comparison** tests where possible
- [ ] **Verify ALL API contracts** produce same responses
- [ ] **Test business logic edge cases** match source behavior
- [ ] **Validate integration flows** work identically
- [ ] **Confirm error handling** produces same error messages
- [ ] **Test performance characteristics** meet or exceed source

### 13.2 Documentation & Handover
- [ ] **Document any intentional differences** from source
- [ ] **Create operational runbooks** for new system
- [ ] **Document troubleshooting guides** for common issues
- [ ] **Create monitoring dashboards** for operational visibility
- [ ] **Prepare rollback procedures** if needed
- [ ] **Train operations team** on new system characteristics

---

## Success Criteria
- [ ] **100% API compatibility** - same responses for same requests
- [ ] **100% business logic parity** - same outputs for same inputs  
- [ ] **Operational readiness** - monitoring, logging, deployment working
- [ ] **Performance parity** - meets or exceeds source performance
- [ ] **Team readiness** - development and operations teams trained

## Migration Complete ✅
Once all TODOs are completed and success criteria met, the migration is ready for production cutover.

---

## APPENDIX A: Diagrams Template (`docs/diagrams.md`)

When creating `docs/diagrams.md`, include these PlantUML diagrams:

### Component Architecture Diagram
```plantuml
@startuml component
!define RECTANGLE class

package "Web Layer" {
  [Controllers] as ctrl
  [Authentication] as auth
  [Middleware] as middleware
}

package "Business Layer" {
  [Services] as services
  [Domain Entities] as entities
  [Validation] as validation
}

package "Data Layer" {
  [Repositories] as repos
  [Database] as db
  [Migrations] as migrations
}

package "Integration Layer" {
  [Message Publisher] as publisher
  [Event Handlers] as handlers
  [External APIs] as apis
}

ctrl --> services
services --> entities
services --> repos
repos --> db
services --> publisher
handlers --> services
@enduml
```

### Database Schema Diagram
```plantuml
@startuml database
!define TABLE class

entity "kodifikaators" as k {
  * id : UUID
  --
  name : VARCHAR(50)
  description : VARCHAR(5000)
  parent_kodifikaator_id : UUID
  root_kodifikaator_id : UUID
  priority : INTEGER
  valid_from : TIMESTAMP
  valid_to : TIMESTAMP
  created_at : TIMESTAMP
  created_by : TEXT
  is_deleted : BOOLEAN
}

entity "kodifikaator_codes" as kc {
  * id : UUID
  --
  kodifikaator_id : UUID
  type_id : UUID
  code : TEXT
  created_at : TIMESTAMP
  is_deleted : BOOLEAN
}

k ||--o{ kc : has
k ||--o{ k : parent_child
@enduml
```

### Domain Entity Relationships
```plantuml
@startuml domain
class Kodifikaator {
  +SetCode(code, typeId)
  +RemoveCode(code)
  +SetCharacteristic(typeId, value)
  +SetLongName(value)
  +SetTranslation(languageId, value)
  +SetReplications(replications)
  +CanDelete() : boolean
}

class KodifikaatorCode {
  +Code : string
  +TypeId : UUID
}

class KodifikaatorCharacteristic {
  +TypeId : UUID
  +Value : string
}

Kodifikaator ||--o{ KodifikaatorCode
Kodifikaator ||--o{ KodifikaatorCharacteristic
Kodifikaator ||--o{ Kodifikaator : parent_children
@enduml
```

### Authentication Flow
```plantuml
@startuml auth
actor User
participant Browser
participant "Auth Controller" as AuthCtrl
participant "OIDC Provider" as OIDC
participant "Security Filter" as Filter

User -> Browser: Access protected resource
Browser -> Filter: HTTP Request
Filter -> Filter: Check for JWT token
alt No token or expired
  Filter -> Browser: 401 Unauthorized
  Browser -> AuthCtrl: GET /api/v1/Auth/Login
  AuthCtrl -> Browser: 302 Redirect to OIDC
  Browser -> OIDC: Authentication flow
  OIDC -> Browser: Redirect with auth code
  Browser -> AuthCtrl: GET /api/v1/Auth/Callback
  AuthCtrl -> OIDC: Exchange code for tokens
  AuthCtrl -> Browser: Set AccessToken cookie
else Valid token
  Filter -> AuthCtrl: Allow request
end
@enduml
```

### API Request Flow
```plantuml
@startuml api_flow
actor Client
participant "Controller" as Ctrl
participant "Service" as Svc
participant "Repository" as Repo
participant "Database" as DB
participant "Message Publisher" as Pub

Client -> Ctrl: POST /api/v1/Kodifikaator
Ctrl -> Ctrl: Validate request
Ctrl -> Svc: create(command)
Svc -> Svc: Business validation
Svc -> Repo: save(entity)
Repo -> DB: INSERT
DB -> Repo: Success
Repo -> Svc: Entity saved
Svc -> Pub: Publish integration event
Pub -> Pub: Store in outbox
Svc -> Ctrl: Return ID
Ctrl -> Client: 201 Created
@enduml
```

### Integration Message Flow
```plantuml
@startuml integration
participant "Kodifikaator Service" as KS
participant "Outbox Processor" as Outbox
participant "RabbitMQ" as MQ
participant "External System" as Ext
database "Outbox Table" as OT

KS -> OT: Store integration event
Outbox -> OT: Poll for pending messages
OT -> Outbox: Return pending events
Outbox -> MQ: Publish message
MQ -> Ext: Route message
Ext -> Ext: Process event
Outbox -> OT: Mark as processed
@enduml
```

---

## APPENDIX B: OpenAPI Generation Rules

When creating `docs/openapi.yml`:

### 1. Exact Property Preservation
- **Preserve exact property names** and casing from source DTOs
- **Map enums exactly** as they appear in source code
- **Include all validation constraints** (required, min/max, patterns)

### 2. Authentication Schemes
```yaml
components:
  securitySchemes:
    oauth2:
      type: oauth2
      flows:
        authorizationCode:
          authorizationUrl: https://auth.provider.com/auth
          tokenUrl: https://auth.provider.com/token
          scopes:
            read: Read access
            write: Write access
    cookieAuth:
      type: apiKey
      in: cookie
      name: AccessToken
```

### 3. Error Response Standards
```yaml
components:
  schemas:
    ProblemDetail:
      type: object
      properties:
        type:
          type: string
          format: uri
        title:
          type: string
        status:
          type: integer
        detail:
          type: string
        instance:
          type: string
          format: uri
```

### 4. Validation Rules
- **Extract from source annotations**: `[Required]`, `[StringLength]`, `[Range]`, etc.
- **Map to OpenAPI constraints**: `required`, `minLength`, `maxLength`, `minimum`, `maximum`
- **Include pattern validation**: regex patterns from source validation

---

---

## 🚨 CRITICAL ANTI-PATTERNS TO AVOID

### **Based on Real Migration Experience - These Gaps Nearly Caused Production Issues:**

1. **🚫 Don't assume string parsing logic is trivial**
   - **Real Example**: X-Road client parsing `"EE/GOV/12345678/service"` 
   - **C# extracts**: First segment `"EE"` 
   - **Java was extracting**: Second segment `"GOV"`
   - **Impact**: Silent data corruption in production

2. **🚫 Don't skip hierarchical propagation logic**
   - **Real Example**: C# `SetReplications` propagates to ancestors AND removes from descendants
   - **Java initially**: Only set on target entity
   - **Impact**: Data inconsistency across hierarchy

3. **🚫 Don't miss conditional integration events**
   - **Real Example**: C# publishes `QueuesSetIntegrationEvent` only for special parent IDs
   - **Java initially**: No special handling  
   - **Impact**: Missing integration events for critical entities

4. **🚫 Don't overlook middleware integration**
   - **Real Example**: Java had TokenIntrospectionMiddleware but wasn't registered
   - **Impact**: Security middleware not functioning

5. **🚫 Don't ignore constant value differences**
   - **Real Example**: Missing domain constants caused method implementations to fail
   - **Impact**: Business logic couldn't function properly

6. **🚫 Don't assume OpenAPI specs are accurate**
   - **Real Example**: OpenAPI documented non-existent endpoints, missing security
   - **Impact**: Implementation didn't match actual C# behavior

7. **🚫 Don't skip publication tracking**
   - **Real Example**: C# tracks all message publications in database
   - **Java initially**: No tracking after publishing
   - **Impact**: No operational visibility, different delete validation

8. **🚫 Don't assume database schemas are complete in initial migration**
   - **Real Example**: Missing audit tables, support tables, schema inconsistencies
   - **Java initially**: Incomplete schema causing foreign key violations
   - **Impact**: Test failures, missing operational data, incorrect constraints

9. **🚫 Don't use placeholder data in production logic**
   - **Real Example**: Publication tracking using fake UUIDs for environment/service IDs
   - **Java initially**: Hardcoded placeholder UUIDs causing constraint violations
   - **Impact**: Test failures, incorrect business logic, data integrity issues

10. **🚫 Don't bypass business logic in tests**
    - **Real Example**: Tests calling helper methods directly instead of service methods
    - **Java initially**: Tests passed but business validation wasn't actually tested
    - **Impact**: Business rules not validated, false confidence in test coverage

11. **🚫 Don't ignore empty/null collection handling in business logic**
    - **Real Example**: C# `SetReplications` processes empty lists to remove replications from descendants
    - **Java initially**: Empty replications list was ignored, descendants kept old replications
    - **Impact**: Hierarchical data inconsistency, replication removal didn't work

12. **🚫 Don't assume test environments match production constraints**
    - **Real Example**: Missing foreign key parent records in test database
    - **Java initially**: Tests failed with constraint violations for missing parent kodifikaators
    - **Impact**: Test failures masking real business logic issues

13. **🚫 Don't overlook JWT claims processing differences**
    - **Real Example**: C# filters out "openid" scope from permissions, Java didn't initially
    - **Java initially**: Included "openid" in permissions list
    - **Impact**: Different authentication behavior, incorrect permission sets

14. **🚫 Don't miss required field validation in update operations**
    - **Real Example**: C# requires `validFrom` field in update commands for business validation
    - **Java initially**: Tests missing `validFrom` causing validation failures
    - **Impact**: Update operations failing validation unexpectedly

15. **🚫 Don't ignore database audit field requirements**
    - **Real Example**: C# inserts include `ID`, `CREATED_AT`, `CREATED_BY`, `UPDATED_AT`, `UPDATED_BY`, `IS_DELETED`
    - **Java initially**: Missing required audit fields in insert statements
    - **Impact**: Database constraint violations, incomplete audit trails

16. **🚫 Don't assume container configurations are environment-agnostic**
    - **Real Example**: TestContainers using public DockerHub image while production uses private registry
    - **Java initially**: Wrong RabbitMQ image causing test container startup failures
    - **Impact**: Test infrastructure failures, different behavior between test and production

### **Validation Checklist - Ask These Questions:**
- [ ] Are we extracting the SAME segment from parsed strings?
- [ ] Do hierarchical operations propagate to ancestors AND descendants?
- [ ] Are integration events published for ALL the same conditions?
- [ ] Are ALL middlewares registered in the correct order?
- [ ] Do we have ALL constants/configuration values from source?
- [ ] Does the API contract match the ACTUAL source implementation?
- [ ] Are we tracking operational data the same way?
- [ ] Does the database schema include ALL tables from the source (especially audit tables)?
- [ ] Are foreign key constraints and data types exactly matching the source?
- [ ] Is production logic using real data instead of placeholder values?
- [ ] **Do tests call actual service methods instead of bypassing business logic?**
- [ ] **Does empty/null collection handling match source behavior exactly?**
- [ ] **Are all required parent/reference records created in test setup?**
- [ ] **Does JWT claims processing filter the same scopes as the source?**
- [ ] **Are all required fields (like `validFrom`) included in command objects?**
- [ ] **Do database inserts include ALL required audit fields?**
- [ ] **Do test containers use the same images/configurations as production?**

## Notes for AI Implementation
When using this guide:
1. **Work through TODOs sequentially** - each builds on previous ones
2. **Verify each step** before moving to next
3. **Test incrementally** - don't wait until end to test
4. **Document deviations** - note any differences from source behavior
5. **Focus on behavior preservation** - not just interface compatibility
6. **Create visual artifacts early** - diagrams help understand complex systems
7. **Validate OpenAPI contract** against actual controller implementations
8. **🚨 PAY SPECIAL ATTENTION TO ANTI-PATTERNS** - these are proven failure points
